<?php
header('Content-Type: application/json; charset=utf-8');
$config = require 'config.php';

// Initialize variables
$order_number = null;
$request_method = $_SERVER['REQUEST_METHOD'];

// Handle different request methods
switch ($request_method) {
    case 'GET':
        $order_number = $_GET['order_number'] ?? null;
        break;
        
    case 'POST':
        $requestBody = file_get_contents('php://input');
        $data = json_decode($requestBody, true);
        $order_number = $data['order_number'] ?? null;
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        exit;
}

// Debug logging
error_log('Request Method: ' . $request_method);
error_log('Order Number: ' . var_export($order_number, true));
error_log('GET params: ' . var_export($_GET, true));

// Enhanced request logging
$logFile = 'request.log';
$logData = [
    'timestamp' => date('Y-m-d H:i:s'),
    'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
    'request_method' => $request_method,
    'request_body' => $request_method === 'POST' ? $requestBody : '',
    'get_params' => $_GET,
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'unknown',
    'request_headers' => getallheaders(),
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
    'request_uri' => $_SERVER['REQUEST_URI'] ?? 'unknown'
];

// Log the request
$logEntry = json_encode($logData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
file_put_contents($logFile, $logEntry . "\n---\n", FILE_APPEND | LOCK_EX);

// Extract order number from URL if it's in the path
if (empty($order_number)) {
    $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $pathParts = explode('/', trim($path, '/'));
    $order_number = end($pathParts);
}

// Validate order number
if (empty($order_number)) {
    $error_response = [
        'error' => 'order_number is required',
        'debug' => [
            'method' => $request_method,
            'uri' => $_SERVER['REQUEST_URI'],
            'get_params' => $_GET,
            'path_info' => $pathParts ?? []
        ]
    ];
    echo json_encode($error_response);
    http_response_code(400);
    exit;
}
// API configuration
$apiUrl = $config['new_api_url_pay_status'];
$apiKey = $config['api_key'];

// Build query string
$queryString = http_build_query(['order_number' => $order_number]);
$fullUrl = $apiUrl . '?' . $queryString;

// Initialize cURL session with enhanced options
$ch = curl_init($fullUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . $apiKey,
        'Accept: application/json',
        'Content-Type: application/json'
    ],
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_SSL_VERIFYHOST => 2,
    CURLOPT_TIMEOUT => 30
]);

// ... existi

// Execute the request
$response = curl_exec($ch);

// Log the API request
$logFile = 'api_request.log';
$logData = date('Y-m-d H:i:s') . " - Order Check Request: " . $fullUrl . "\n";
file_put_contents($logFile, $logData, FILE_APPEND | LOCK_EX);

if (curl_errno($ch)) {
    echo json_encode(['error' => curl_error($ch)]);
    http_response_code(500);
} else {
    $responseData = json_decode($response, true);
    
    // Log the response
    error_log('Raw API Response: ' . $response);
    
    if (isset($responseData['code'])) {
        if ($responseData['code'] === 0) {
            // Success response
            echo json_encode([
                'success' => true,
                'data' => [
                    'number' => $responseData['data']['number'] ?? null,
                    'order_number' => $responseData['data']['order_number'] ?? null,
                    'channel_name' => $responseData['data']['channel_name'] ?? null,
                    'provider_type' => $responseData['data']['provider_type'] ?? null,
                    'currency' => $responseData['data']['currency'] ?? null,
                    'amount' => $responseData['data']['amount'] ?? null,
                    'fee' => $responseData['data']['fee'] ?? null,
                    'status' => $responseData['data']['status'] ?? null,
                    'callback_status' => $responseData['data']['callback_status'] ?? null,
                    'payee_name' => $responseData['data']['payee_name'] ?? null,
                    'payee_bank_code' => $responseData['data']['payee_bank_code'] ?? null,
                    'payee_bank_account' => $responseData['data']['payee_bank_account'] ?? null,
                    'created_at' => $responseData['data']['created_at'] ?? null,
                    'updated_at' => $responseData['data']['updated_at'] ?? null,
                    'expired_at' => $responseData['data']['expired_at'] ?? null
                ]
            ]);
            http_response_code(200);
        
        } else {
            // Error response
            echo json_encode([
                'error' => $responseData['message'] ?? 'Unknown error',
                'code' => $responseData['code']
            ]);
            http_response_code(400);
        }
    } else {
        echo json_encode(['error' => 'Malformed API response']);
        http_response_code(500);
    }
}

curl_close($ch);
?>